<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Commande;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Commande::with('user');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                  ->orWhere('name_client', 'like', "%{$search}%")
                  ->orWhere('prenom_client', 'like', "%{$search}%")
                  ->orWhereHas('user', function($qu) use ($search) {
                      $qu->where('nom', 'like', "%{$search}%")
                         ->orWhere('prenom', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('statut')) {
            $query->where('statut', $request->statut);
        }

        $commandes = $query->orderBy('created_at', 'desc')
            ->paginate(20)
            ->withQueryString();

        return view('admin.orders.index', compact('commandes'));
    }

    public function show($id)
    {
        $commande = Commande::with(['user', 'products'])->findOrFail($id);
        $shop = \App\Models\User::where('role', 'admin')->first();
        return view('admin.orders.show', compact('commande', 'shop'));
    }

    public function updateStatus(Request $request, $id)
    {
        $commande = Commande::with('products')->findOrFail($id);
        $old_status = $commande->statut;

        $validated = $request->validate([
            'statut' => 'required|in:en_attente,en_cours,livres,annulee'
        ]);

        $new_status = $validated['statut'];

        if ($old_status === $new_status) {
            return redirect()->back();
        }

        \DB::beginTransaction();
        try {
            // ========== STOCK MANAGEMENT LOGIC ==========
            $in_stock_states = ['en_attente', 'annulee'];
            $sold_states = ['en_cours', 'livres'];

            // DEDUCT stock: In Stock -> Sold
            if (in_array($old_status, $in_stock_states) && in_array($new_status, $sold_states)) {
                foreach ($commande->products as $product) {
                    $product->decrement('en_stock', $product->pivot->quantite);
                }
            }
            // RESTORE stock: Sold -> In Stock
            elseif (in_array($old_status, $sold_states) && in_array($new_status, $in_stock_states)) {
                foreach ($commande->products as $product) {
                    $product->increment('en_stock', $product->pivot->quantite);
                }
            }
            // Transitions within the same group (e.g., en_cours -> livres) or (en_attente -> annulee) 
            // do not change stock levels as they are already accounted for.

            $commande->update($validated);
            \DB::commit();

            return redirect()->back()->with('success', 'Statut mis à jour et stock actualisé avec succès!');
        } catch (\Exception $e) {
            \DB::rollBack();
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la mise à jour du statut.');
        }
    }

    public function destroy($id)
    {
        $commande = Commande::with('products')->findOrFail($id);
        
        \DB::beginTransaction();
        try {
            // Restore stock if the order was processed (en_cours or livres status)
            if (in_array($commande->statut, ['en_cours', 'livres'])) {
                foreach ($commande->products as $product) {
                    $product->increment('en_stock', $product->pivot->quantite);
                }
            }
            
            // Delete order items first (due to foreign key constraints)
            $commande->products()->detach();
            
            // Delete the order
            $commande->delete();
            
            \DB::commit();
            
            return redirect()->route('admin.orders.index')->with('success', 'Commande supprimée et stock restauré avec succès!');
        } catch (\Exception $e) {
            \DB::rollBack();
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la suppression de la commande.');
        }
    }
}
