<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    public function index(Request $request, $category_slug = null)
    {
        $query = Product::query();
        $category = null;

        // 1. Determine Context (Category)
        if ($request->filled('category_id')) {
            $category = Category::find($request->category_id);
            if ($category) {
                $query->where('category_id', $category->id);
            }
        } elseif ($category_slug) {
            $category = Category::where('slug', $category_slug)->firstOrFail();
            $query->where('category_id', $category->id);
        }

        // 2. Prepare Facet Query (Base Context for counts)
        // We use this to generate the sidebar options based on available products in this category
        $facetQuery = clone $query;

        // 3. Apply Filters to Main Query
        
        // Price Range
        if ($request->filled('min_price')) {
            $query->where('prix_actuel', '>=', $request->min_price);
        }
        if ($request->filled('max_price')) {
            $query->where('prix_actuel', '<=', $request->max_price);
        }

        // Multiple Selection Filters (Arrays)
        if ($request->filled('marque')) {
            $query->whereIn('marque', (array)$request->marque);
        }
        if ($request->filled('diametre')) {
            $query->whereIn('diametre', (array)$request->diametre);
        }
        if ($request->filled('hauteur')) {
            $query->whereIn('hauteur', (array)$request->hauteur);
        }
        if ($request->filled('largeur')) {
            $query->whereIn('largeur', (array)$request->largeur);
        }
        if ($request->filled('charge')) {
            $query->whereIn('charge_index', (array)$request->charge);
        }
        if ($request->filled('vitesse')) {
            $query->whereIn('vitesse_index', (array)$request->vitesse);
        }
        if ($request->filled('saison')) {
            $query->whereIn('saison', (array)$request->saison);
        }
        if ($request->filled('runflat')) {
            $query->whereIn('runflat', (array)$request->runflat);
        }
        
        // Sorting
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'price_asc': $query->orderBy('prix_actuel', 'asc'); break;
            case 'price_desc': $query->orderBy('prix_actuel', 'desc'); break;
            case 'name_asc': $query->orderBy('name', 'asc'); break;
            default: $query->latest(); break;
        }

        $products = $query->paginate(9)->withQueryString();
        
        // 4. Calculate Facets (Counts) using $facetQuery
        // This ensures specific filter options are relevant to the current category

        $marques = (clone $facetQuery)->select('marque', DB::raw('count(*) as count'))
            ->groupBy('marque')->orderBy('marque')->get();

        $diametres = (clone $facetQuery)->select('diametre', DB::raw('count(*) as count'))
            ->whereNotNull('diametre')->groupBy('diametre')->orderBy('diametre')->get();

        $hauteurs = (clone $facetQuery)->select('hauteur', DB::raw('count(*) as count'))
            ->whereNotNull('hauteur')->groupBy('hauteur')->orderBy('hauteur')->get();

        $largeurs = (clone $facetQuery)->select('largeur', DB::raw('count(*) as count'))
            ->whereNotNull('largeur')->groupBy('largeur')->orderBy('largeur')->get();

        $charges = (clone $facetQuery)->select('charge_index', DB::raw('count(*) as count'))
            ->whereNotNull('charge_index')->groupBy('charge_index')->orderBy('charge_index')->get();

        $vitesses = (clone $facetQuery)->select('vitesse_index', DB::raw('count(*) as count'))
            ->whereNotNull('vitesse_index')->groupBy('vitesse_index')->orderBy('vitesse_index')->get();

        $saisons = (clone $facetQuery)->select('saison', DB::raw('count(*) as count'))
            ->whereNotNull('saison')->groupBy('saison')->orderBy('saison')->get();

        $runflats = (clone $facetQuery)->select('runflat', DB::raw('count(*) as count'))
            ->whereNotNull('runflat')->groupBy('runflat')->orderBy('runflat', 'desc')->get();

        // Price Range for Slider limits
        $priceStats = (clone $facetQuery)->selectRaw('MIN(prix_actuel) as min_price, MAX(prix_actuel) as max_price')->first();
        $minPriceLimit = $priceStats->min_price ?? 0;
        $maxPriceLimit = $priceStats->max_price ?? 10000;

        // Get all categories for the filter dropdown
        $categories = Category::all();

        return view('pneus.index', compact(
            'products', 'category', 'categories',
            'marques', 'diametres', 'hauteurs', 'largeurs', 
            'charges', 'vitesses', 'saisons', 'runflats',
            'minPriceLimit', 'maxPriceLimit'
        ));
    }

    public function show($id)
    {
        $product = Product::with('category')->findOrFail($id);
        return view('pneus.show', compact('product'));
    }
}
